﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using System.Threading.Tasks;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Cors;
using Microsoft.AspNetCore.Hosting;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using Sonic.Services;
using Sonic.Services.Controllers;
using Sonic.Web.DAL;
using Sonic.Web.Model;
using Sonic.Web.Resources;
using Sonic.Web.Service;

namespace Sonic.Service.PL.Controllers.Echo
{
    [Route("api/[controller]/[action]")]
    [ApiController]
    [Authorize]
    public class EchoOrderController : BaseController
    {
        private readonly OrderService _orderService;
        private readonly PromotionService _promotionService;
        private readonly IRequestRepository _requestRepository;
        private IHostingEnvironment _hostingEnvironment;
    

        public EchoOrderController(IHostingEnvironment hostingEnvironment, IRequestRepository requestRepository) : base(requestRepository)
        {
            _requestRepository = requestRepository;
            _orderService = new OrderService(_requestRepository);
            _promotionService = new PromotionService(_requestRepository);
            _hostingEnvironment = hostingEnvironment;
        }

        [HttpPost]
        [ActionName("SaveOrder")]
        public JsonResult SaveOrder([FromBody] List<TransactionCommonData> trans)
        {
            try
            {
                if (!ModelState.IsValid)
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
                var result = _orderService.SaveOrderEcho(ref trans);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessJSon(trans, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception ex)
            {
                ErrorLogger.Logger(MethodBase.GetCurrentMethod().DeclaringType.FullName, System.Reflection.MethodBase.GetCurrentMethod().Name, ex.Message, ex.InnerException, ex.StackTrace, 0);
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpPost]
        [ActionName("CheckOutOrder")]
        public JsonResult CheckOutOrder([FromBody] List<ItemPackModel> soldItems)
        {
            GlobalErrors result = GlobalErrors.Success;
            try
            {
                List<TransactionCommonData> trans = new List<TransactionCommonData>();
                string closeToPromotionIds = string.Empty;
                result = _orderService.CheckOutOrderForEcho(_hostingEnvironment.ContentRootPath, soldItems, ref trans, ref closeToPromotionIds);
                if (result == GlobalErrors.Success)
                {
                    var data = new
                    {
                        trans = trans,
                        closeToPromotionIds = closeToPromotionIds
                    };
                    return JSonResultResponse.GetSuccessJSon(data, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception)
            {
                return JSonResultResponse.GetErrorJSon();
            }
        }

        [HttpGet]
        [ActionName("GetOrderHistory")]
        public JsonResult GetOrderHistory()
        {
            GlobalErrors result = GlobalErrors.Success;
            try
            {
                List<EchoOrderModel> list = new List<EchoOrderModel>();
                result = _orderService.GetCustomerOrderHistoryForEcho(ref list);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessJSon(list, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception)
            {
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpGet]
        [ActionName("GetOrderDetails")]
        public JsonResult GetOrderDetails(string orderId, bool getStatus)
        {
            GlobalErrors result = GlobalErrors.Success;
            try
            {
                List<ItemPackModel> soldItems = new List<ItemPackModel>();
                List<Promotion> achivements = new List<Promotion>();
                List<Promotion> applied = new List<Promotion>();
                List<ItemPackModel> promotedItems = new List<ItemPackModel>();
                decimal netTotal = 0;
                decimal discountTotal = 0; 
                decimal taxTotal = 0;
                decimal grossTotal = 0;
                int statusId = 0;
                string notes = string.Empty;
                result = _orderService.GetOrderDataForEcho(orderId, getStatus, ref soldItems, ref promotedItems, ref statusId, ref notes, ref netTotal, ref grossTotal, ref discountTotal, ref taxTotal, ref achivements, ref applied);
                if (result == GlobalErrors.Success)
                {
                    var data = new
                    {
                        soldItems = soldItems,
                        promotedItems = promotedItems,
                        netTotal = netTotal,
                        grossTotal = grossTotal,
                        discountTotal = discountTotal,
                        taxTotal = taxTotal,
                        statusId = statusId,
                        notes = notes,
                        achivements = achivements,
                        applied = applied
                    };
                    return JSonResultResponse.GetSuccessJSon(data, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception)
            {
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpGet]
        [ActionName("GetOrderDetailsForReorder")]
        public JsonResult GetOrderDetailsForReorder(string orderId)
        {
            GlobalErrors result = GlobalErrors.Success;
            try
            {
                List<ItemPackModel> soldItems = new List<ItemPackModel>();
                ItemService itemService = new ItemService(_requestRepository);
                result = _orderService.GetOrderDetailsForEchoReorder(orderId, ref soldItems, false);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessJSon(soldItems, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception)
            {
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpGet]
        [ActionName("GetOrderDetailsForReorderNew")]
        public JsonResult GetOrderDetailsForReorderNew(string orderId)
        {
            GlobalErrors result = GlobalErrors.Success;
            try
            {
                List<ItemPackModel> soldItems = new List<ItemPackModel>();
                ItemService itemService = new ItemService(_requestRepository);
                result = _orderService.GetOrderDetailsForEchoReorder(orderId, ref soldItems, true);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessJSon(soldItems, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception)
            {
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("CancelOrder")]
        public JsonResult CancelOrder([FromBody] EchoOrderModel order)
        {
            GlobalErrors result = GlobalErrors.Success;
            try
            {
                result = _orderService.CancelOrderForEcho(order);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception)
            {
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpGet]
        [ActionName("GetMyPromotions")]
        public JsonResult GetMyPromotions()
        {
            GlobalErrors result = GlobalErrors.Success;
            try
            {
                List<Promotion> customerPromotions = new List<Promotion>();
                result = _promotionService.EchoGeneratePromotionFile(_hostingEnvironment.ContentRootPath, ref customerPromotions);
                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessJSon(customerPromotions, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception)
            {
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpPost]
        [ActionName("UpdateOrderStatus")]
        public JsonResult UpdateOrderStatus([FromBody] EchoOrderModel order)
        {
            GlobalErrors result = GlobalErrors.Success;
            try
            {
                if (order.StatusId == OrderStatus.Canceled.GetHashCode())
                    result = _orderService.CancelOrderForEcho(order);
                else if(order.StatusId == OrderStatus.New.GetHashCode())
                    result = _orderService.ApproveOrderForEcho(order);

                if (result == GlobalErrors.Success)
                {
                    return JSonResultResponse.GetSuccessJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception)
            {
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpGet]
        [ActionName("IsEchoFirstOrderCustomerOutlet")]
        public JsonResult IsEchoFirstOrderCustomerOutlet()
        {
            GlobalErrors result = GlobalErrors.Success;
            bool isFirstOrderCustomerOutlet = false;
            bool isEchoFirstOrderHaveFreeItems = false;
            try
            {
                result = _orderService.IsEchoFirstOrderCustomerOutlet(ref isFirstOrderCustomerOutlet, ref isEchoFirstOrderHaveFreeItems);
                if (result == GlobalErrors.Success)
                {
                    var data = new
                    {
                        isFirstOrderCustomerOutlet = isFirstOrderCustomerOutlet,
                        isEchoFirstOrderHaveFreeItems = isEchoFirstOrderHaveFreeItems
                    };
                    string msg = ResourcesManager.TranslateKey(MessagesConstants.Desc_Msg_Echo_First_Discount, _requestRepository.LanguageId);
                    msg = string.Format(msg, _requestRepository.Configurations.EchoFirstDiscountCustomerOutlet);
                    return JSonResultResponse.GetSuccessJSon(data, msg);
                }
                else
                {
                    return JSonResultResponse.GetErrorJSon(ResourcesManager.TranslateKey(MessagesConstants.Desc_Error_While_Getting_Data, _requestRepository.LanguageId));
                }
            }
            catch (Exception)
            {
                return JSonResultResponse.GetErrorJSon();
            }
        }
        [HttpGet]
        [ActionName("GetEchoFirstOrderFreeItems")]
        public JsonResult GetEchoFirstOrderFreeItems()
        {
            GlobalErrors result = GlobalErrors.Success;
            List<ItemPackModel> itemPacks;
            try
            {
                itemPacks = new List<ItemPackModel>();
                result = _orderService.GetEchoFirstOrderFreeItems(ref itemPacks);
                return JSonResultResponse.GetSuccessJSon(itemPacks, ResourcesManager.TranslateKey(MessagesConstants.Desc_Success_Getting_Data, _requestRepository.LanguageId));

            }
            catch (Exception)
            {
                return JSonResultResponse.GetErrorJSon();
            }
        }
    }
}